--[[
	Lock Steering Axles

	This script allows you to lock the steering axles of a vehicle.

	@author: 		[LSFM] BayernGamers
	@contributors: 	ifko[nator]
	@date: 			07.11.2024
	@version:		4.0
	
	History:	v1.0 @07.05.2017 - initial implementation in FS 17
				------------------------------------------------------------------------------------------------
				v1.5 @10.05.2017 - add save function and support for the old wheel entry
				------------------------------------------------------------------------------------------------
									if the steering axle is unlocked, it will steer also in backwards direction
				------------------------------------------------------------------------------------------------
				v1.6 @13.05.2017 - fix for MP
				------------------------------------------------------------------------------------------------
				v1.7 @20.08.2017 - fix for joining players
				------------------------------------------------------------------------------------------------
				v1.8 @13.09.2017 - again fix for MP
				------------------------------------------------------------------------------------------------
				v2.0 @04.01.2019 - initial implementation in FS 19
				------------------------------------------------------------------------------------------------
				V2.1 @13.01.2019 - bug fix for BagLifter-Mod
				------------------------------------------------------------------------------------------------
				V2.2 @08.03.2019 - add support for trailers with an steering axle target node
				------------------------------------------------------------------------------------------------
				v2.3 @04.08.2021 - add automatic lock for the steering axles, if AutoDrive is driving backwards
				------------------------------------------------------------------------------------------------
				v3.0 @02.01.2022 - initial implementation in FS 22
				------------------------------------------------------------------------------------------------
				v3.1 @04.01.2022 - fix issues with the nardi cutter trailers
				------------------------------------------------------------------------------------------------
				v3.3 @16.03.2022 - fix for MP
				------------------------------------------------------------------------------------------------
				v3.4 @21.04.2022 - fix for patch 1.4 and higher
				------------------------------------------------------------------------------------------------
				v4.0 @07.11.2024 - convert to FS25

	License: 	This work is licensed under the Creative Commons Attribution-NoDerivs 4.0 International License (CC BY-ND 4.0).

                    Terms:
                        Attribution:
                            You must give appropriate credit to the original author when using this work.
                        No Derivatives:
                            You may not alter, transform, or build upon this work in any way.
                        Usage: 
                            The work may be used for personal and commercial purposes, provided it is not modified or adapted.

                        Additional Clause:
                            This script may not be converted, adapted, or incorporated into any other game versions or platforms except by GIANTS Software.

                        Full License Text:
                            The complete license text can be found at: https://creativecommons.org/licenses/by-nd/4.0/
]]

source(Utils.getFilename("scripts/utils/LoggingUtil.lua", g_currentModDirectory))
source(Utils.getFilename("scripts/utils/AdditionalMathUtil.lua", g_currentModDirectory))
source(Utils.getFilename("scripts/LockSteeringAxlesEvent.lua", g_currentModDirectory))

local log = LoggingUtil

LockSteeringAxles = {}
LockSteeringAxles.currentModName = g_currentModName
LockSteeringAxles.currentModDirectory = g_currentModDirectory

function LockSteeringAxles.initSpecialization()
	local schemaSavegame = Vehicle.xmlSchemaSavegame

	schemaSavegame:register(XMLValueType.BOOL, "vehicles.vehicle(?)." .. LockSteeringAxles.currentModName .. ".lockSteeringAxles#lockSteeringAxle", "Steering Axle is locked.", false)
	log.printDevInfo("Registered 'lockSteeringAxles' xmlPaths", 1, true, "LockSteeringAxles.lua")
end

function LockSteeringAxles.prerequisitesPresent(specializations)
	return SpecializationUtil.hasSpecialization(Attachable, specializations)
end

function LockSteeringAxles.registerEventListeners(vehicleType)
	SpecializationUtil.registerEventListener(vehicleType, "onLoad", LockSteeringAxles)
	SpecializationUtil.registerEventListener(vehicleType, "onUpdate", LockSteeringAxles)
	SpecializationUtil.registerEventListener(vehicleType, "saveToXMLFile", LockSteeringAxles)
	SpecializationUtil.registerEventListener(vehicleType, "onWriteStream", LockSteeringAxles)
	SpecializationUtil.registerEventListener(vehicleType, "onReadStream", LockSteeringAxles)
	SpecializationUtil.registerEventListener(vehicleType, "onRegisterActionEvents", LockSteeringAxles)
end

function LockSteeringAxles.registerFunctions(vehicleType)
	SpecializationUtil.registerFunction(vehicleType, "setSteeringAxleActive", LockSteeringAxles.setSteeringAxleActive)
end

function LockSteeringAxles:onLoad(savegame)
	self.spec_lockSteeringAxles = {}
	local spec_lockSteeringAxles = self.spec_lockSteeringAxles
	local spec_wheels = self.spec_wheels
	local spec_attachable = self.spec_attachable

	spec_lockSteeringAxles.foundSteeringAxle = false
	spec_lockSteeringAxles.lockSteeringAxle = false

	spec_lockSteeringAxles.l10nTexts = {}
	
	spec_lockSteeringAxles.l10nTexts["LOCK_STEERING_AXLE"] = g_i18n:getText("LOCK_STEERING_AXLE", LockSteeringAxles.currentModName)
	spec_lockSteeringAxles.l10nTexts["UNLOCK_STEERING_AXLE"] = g_i18n:getText("UNLOCK_STEERING_AXLE", LockSteeringAxles.currentModName)

	if spec_wheels == nil or #spec_wheels.wheels == 0 or spec_attachable.steeringAxleTargetNode ~= nil or spec_attachable.steeringAxleReferenceComponentNode ~= nil then
		return
	end
	
	if self.xmlFile:hasProperty("vehicle.attachable.steeringAxleAngleScale") then
		spec_lockSteeringAxles.foundSteeringAxle = true
			
		spec_attachable.steeringAxleUpdateBackwards = true
	end

	if spec_lockSteeringAxles.foundSteeringAxle then
		if savegame ~= nil then
			log.printDevInfo("Loading 'lockSteeringAxles' xmlPaths '" .. savegame.key .. "#lockSteeringAxle'", 1, true, "LockSteeringAxles.lua")
			spec_lockSteeringAxles.lockSteeringAxle = savegame.xmlFile:getValue(savegame.key .. "." .. LockSteeringAxles.currentModName .. ".lockSteeringAxles#lockSteeringAxle", spec_lockSteeringAxles.lockSteeringAxle)
			log.printDevInfo("lockSteeringAxle: " .. tostring(spec_lockSteeringAxles.lockSteeringAxle), 1, true, "LockSteeringAxles.lua")
		else
			spec_lockSteeringAxles.lockSteeringAxle = false
		end
		
		self:setSteeringAxleActive(spec_lockSteeringAxles.lockSteeringAxle, false)
	end
end

function LockSteeringAxles:onWriteStream(streamId, connection)
	if not connection:getIsServer() then
		local spec_lockSteeringAxles = self.spec_lockSteeringAxles
		
		if spec_lockSteeringAxles.foundSteeringAxle and spec_lockSteeringAxles.lockSteeringAxle ~= nil then
			streamWriteBool(streamId, spec_lockSteeringAxles.lockSteeringAxle)
		end
	end
end

function LockSteeringAxles:onReadStream(streamId, connection)
	if connection:getIsServer() then
		local spec_lockSteeringAxles = self.spec_lockSteeringAxles
		
		if spec_lockSteeringAxles.foundSteeringAxle and spec_lockSteeringAxles.lockSteeringAxle ~= nil then	
			spec_lockSteeringAxles.lockSteeringAxle = streamReadBool(streamId)
			
			self:setSteeringAxleActive(spec_lockSteeringAxles.lockSteeringAxle, true)
		end
	end
end

function LockSteeringAxles:onRegisterActionEvents(isActiveForInput)
	local spec_lockSteeringAxles = self.spec_lockSteeringAxles
	
	if self.isClient and spec_lockSteeringAxles.foundSteeringAxle then
		spec_lockSteeringAxles.actionEvents = {}       
		self:clearActionEventsTable(spec_lockSteeringAxles.actionEvents)

		if self:getIsActiveForInput(true) then
            local actionEventId
            
			_, actionEventId = self:addActionEvent(spec_lockSteeringAxles.actionEvents, InputAction.TOGGLE_LOCK_STEERING_AXLE_BUTTON, self, LockSteeringAxles.toggleSteeringAxleActive, false, true, false, true, nil)

			spec_lockSteeringAxles.actionEventId = actionEventId

			g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_NORMAL)
			g_inputBinding:setActionEventTextVisibility(actionEventId, true)
			g_inputBinding:setActionEventActive(actionEventId, false)
		end
	end
end

function LockSteeringAxles.toggleSteeringAxleActive(self, actionName, inputValue, callbackState, isAnalog)
	local spec_lockSteeringAxles = self.spec_lockSteeringAxles
	
	self:setSteeringAxleActive(not spec_lockSteeringAxles.lockSteeringAxle, false)
end

function LockSteeringAxles:onUpdate(dt, isActiveForInput, isSelected)
	local spec_lockSteeringAxles = self.spec_lockSteeringAxles
	local spec_attachable = self.spec_attachable

	if spec_lockSteeringAxles.foundSteeringAxle then
		if self.isClient then	
			spec_attachable.updateSteeringAxleAngle = not spec_lockSteeringAxles.lockSteeringAxle
						
			if spec_lockSteeringAxles.actionEventId ~= nil and self:getIsActive() then
				local currentText = spec_lockSteeringAxles.l10nTexts.LOCK_STEERING_AXLE

				g_inputBinding:setActionEventActive(spec_lockSteeringAxles.actionEventId, true)
				
				if spec_lockSteeringAxles.lockSteeringAxle then
					currentText = spec_lockSteeringAxles.l10nTexts.UNLOCK_STEERING_AXLE
				end
				
				g_inputBinding:setActionEventText(spec_lockSteeringAxles.actionEventId, currentText)
			end
		end

		if spec_lockSteeringAxles.lockSteeringAxle then
			if spec_attachable.steeringAxleAngle ~= 0 then	
				if spec_attachable.steeringAxleAngle > 0 then
					spec_attachable.steeringAxleAngle = math.max(spec_attachable.steeringAxleAngle - spec_attachable.steeringAxleAngleSpeed / 2 * dt, 0)
				else
					spec_attachable.steeringAxleAngle = math.min(spec_attachable.steeringAxleAngle + spec_attachable.steeringAxleAngleSpeed / 2 * dt, 0)
				end
			end
		end
	end

	if spec_attachable.updateSteeringAxleAngle then
		local steeringAngle = 0
		local baseVehicle = self:getSteeringAxleBaseVehicle()

		if (baseVehicle ~= nil or spec_attachable.steeringAxleReferenceComponentNode ~= nil) and (self.movingDirection >= 0 or spec_attachable.steeringAxleUpdateBackwards) then
			yRot = Utils.getYRotationBetweenNodes(self.steeringAxleNode, spec_attachable.steeringAxleReferenceComponentNode or baseVehicle.steeringAxleNode)

			local scale = 1

			if spec_attachable.steeringAxleAngleScaleSpeedDependent then
				local startSpeed = spec_attachable.steeringAxleAngleScaleStart
				local endSpeed = spec_attachable.steeringAxleAngleScaleEnd

				scale = AdditionalMathUtil.clamp(1 + (self:getLastSpeed() - startSpeed) * 1 / (startSpeed - endSpeed), 0, 1)
			end

			steeringAngle = yRot * scale
		elseif self:getLastSpeed() > 0.2 then
			steeringAngle = 0
		end

		if not self:getIsSteeringAxleAllowed() then
			steeringAngle = 0
		end

		if spec_attachable.steeringAxleDistanceDelay > 0 then
			spec_attachable.steeringAxleTargetAngleHistoryMoved = spec_attachable.steeringAxleTargetAngleHistoryMoved + self.lastMovedDistance

			if spec_attachable.steeringAxleTargetAngleHistoryMoved > 0.1 then
				spec_attachable.steeringAxleTargetAngleHistory[spec_attachable.steeringAxleTargetAngleHistoryIndex] = steeringAngle
				spec_attachable.steeringAxleTargetAngleHistoryIndex = spec_attachable.steeringAxleTargetAngleHistoryIndex + 1

				if spec_attachable.steeringAxleTargetAngleHistoryIndex > #spec_attachable.steeringAxleTargetAngleHistory then
					spec_attachable.steeringAxleTargetAngleHistoryIndex = 1
				end
			end

			local lastIndex = spec_attachable.steeringAxleTargetAngleHistoryIndex + 1

			if lastIndex > #spec_attachable.steeringAxleTargetAngleHistory then
				lastIndex = 1
			end

			spec_attachable.steeringAxleTargetAngle = spec_attachable.steeringAxleTargetAngleHistory[lastIndex]
		else
			spec_attachable.steeringAxleTargetAngle = steeringAngle
		end

		local dir = AdditionalMathUtil.sign(spec_attachable.steeringAxleTargetAngle - spec_attachable.steeringAxleAngle)
		local speed = spec_attachable.steeringAxleAngleSpeed

		if not self.finishedFirstUpdate then
			speed = 9999
		end

		if dir == 1 then
			spec_attachable.steeringAxleAngle = math.min(spec_attachable.steeringAxleAngle + dir * dt * speed, spec_attachable.steeringAxleTargetAngle)
		else
			spec_attachable.steeringAxleAngle = math.max(spec_attachable.steeringAxleAngle + dir * dt * speed, spec_attachable.steeringAxleTargetAngle)
		end

		if spec_attachable.steeringAxleTargetNode ~= nil and (self:getLastSpeed() > 0.25 or not self.finishedFirstUpdate) then
			local angle = nil

			if spec_attachable.steeringAxleTargetNodeRefAngle ~= nil then
				local alpha = AdditionalMathUtil.clamp(spec_attachable.steeringAxleAngle / spec_attachable.steeringAxleTargetNodeRefAngle, -1, 1)

				if alpha >= 0 then
					angle = spec_attachable.steeringAxleAngleMaxRot * alpha
				else
					angle = spec_attachable.steeringAxleAngleMinRot * -alpha
				end
			else
				angle = AdditionalMathUtil.clamp(spec_attachable.steeringAxleAngle, spec_attachable.steeringAxleAngleMinRot, spec_attachable.steeringAxleAngleMaxRot)
			end

			setRotation(spec_attachable.steeringAxleTargetNode, 0, angle * spec_attachable.steeringAxleDirection, 0)

			self:setMovingToolDirty(spec_attachable.steeringAxleTargetNode)
		end
	end
end

function LockSteeringAxles:saveToXMLFile(xmlFile, key, usedModNames)
	local spec_lockSteeringAxles = self.spec_lockSteeringAxles
	
	if spec_lockSteeringAxles.foundSteeringAxle then
		log.printDevInfo("Saving 'lockSteeringAxles' xmlPaths '" .. key .. "#lockSteeringAxle'", 1, true, "LockSteeringAxles.lua")
		log.printDevInfo("lockSteeringAxle: " .. tostring(spec_lockSteeringAxles.lockSteeringAxle), 1, true, "LockSteeringAxles.lua")
		xmlFile:setValue(key .. "#lockSteeringAxle", spec_lockSteeringAxles.lockSteeringAxle)
	end
end

function LockSteeringAxles:setSteeringAxleActive(lockSteeringAxle, noEventSend)
	local spec_lockSteeringAxles = self.spec_lockSteeringAxles
	
	if lockSteeringAxle ~= spec_lockSteeringAxles.lockSteeringAxle then
		spec_lockSteeringAxles.lockSteeringAxle = lockSteeringAxle
		
		if noEventSend == nil or noEventSend == false then
			if g_server ~= nil then
				g_server:broadcastEvent(LockSteeringAxlesEvent.new(self, lockSteeringAxle), nil, nil, self)
			else
				g_client:getServerConnection():sendEvent(LockSteeringAxlesEvent.new(self, lockSteeringAxle))
			end
		end
	end
end